/*
 * <:copyright-BRCM:2017:DUAL/GPL:standard 
 * 
 *    Copyright (c) 2017 Broadcom 
 *    All Rights Reserved
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2, as published by
 * the Free Software Foundation (the "GPL").
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * 
 * A copy of the GPL is available at http://www.broadcom.com/licenses/GPLv2.php, or by
 * writing to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 * 
 * :>
 */

#ifndef _DPI_LOCAL_H
#define _DPI_LOCAL_H

#include <linux/ip.h>
#include <linux/if_ether.h>
#include <linux/dpi.h>
#include <net/netfilter/nf_conntrack_acct.h>
#include <net/netfilter/nf_conntrack_ecache.h>

#include <bcmdpi.h>
#include <tdts.h>

#define DPI_URL_RECORD

struct dpi_classify_parms {
	struct sk_buff		*skb;
	struct nf_conn		*ct;

	struct dpi_appinst	*appinst;
	struct dpi_dev		*dev;
	struct dpi_app		*app;
	struct dpi_url		*url;

	tdts_pkt_parameter_t	*pkt_param;
	unsigned long		flags;
	unsigned long		old_flags;
	int			lookup_flags;
	u8			is_multicast;

	/* status */
	int	nfct_update;
	bool	ct_stats_available;
};

struct dpi_ip {
	union {
		struct in_addr	ip;
		struct in6_addr	ip6;
	};
	u8			l3proto;
	u8			mac[ETH_ALEN];

	struct hlist_node	node;
};

struct dpi_classification_stats {
	u32 lookups;
	u32 hits;
	u32 misses;
	u32 classified;
	u32 unclassified;
};

struct dpi_stats {
	struct dpi_classification_stats apps;
	struct dpi_classification_stats devs;
	struct dpi_classification_stats urls;
	u32 total_lookups;
	u32 app_count;
	u32 dev_count;
	u32 appinst_count;
	u32 url_count;
	u32 engine_errors;
	u64 blocked_pkts;
};

struct proc_dir_entry;

extern struct dpi_stats dpi_stats;
extern struct proc_dir_entry *dpi_dir;

/* dpi tables */
struct dpi_dev *dpi_dev_find_or_alloc(u8 *mac);
struct dpi_app *dpi_app_find_or_alloc(u32 app_id);
struct dpi_appinst *dpi_appinst_find_or_alloc(struct dpi_app *app,
					      struct dpi_dev *dev);
struct dpi_url *dpi_url_find_or_alloc(char *hostname, int len);
struct dpi_ip *dpi_ip_find(u8 *ip, int l3proto);
struct dpi_ip *dpi_ip_find_by_skb(struct sk_buff *skb);
struct dpi_ip *dpi_ip_alloc(u8 *ip, int l3proto, u8* mac);
void dpi_ip_free(struct dpi_ip *entry);
void dpi_ct_destroy(struct nf_conn *ct);
void dpi_reset_stats(unsigned long flags);
int __init dpi_init_tables(void);
void __exit dpi_deinit_tables(void);

#if IS_ENABLED(CONFIG_BCM_DPI_TEST)
uint32_t dpi_classify_app_test(struct sk_buff *skb, unsigned long *flags,
			       int lookup_flags);
int __init dpi_test_init(void);
void dpi_test_exit(void);
#else
static inline uint32_t
dpi_classify_app_test(struct sk_buff *skb, unsigned long *flags,
		      int lookup_flags)
{
	return 0;
}
static inline int __init dpi_test_init(void) { return 0; }
static inline void dpi_test_exit(void) { }
#endif

/* parsing */
void dpi_parse_dhcp(struct dpi_classify_parms *p);

/* helper functions */
static inline struct dpi_info *dpi_info(struct sk_buff *skb)
{
	enum ip_conntrack_info ctinfo;
	struct nf_conn *ct;

	ct = nf_ct_get(skb, &ctinfo);
	if (!ct)
		return NULL;

	return &ct->bcm_ext.dpi;
}

static inline int
cf_l3(struct sk_buff *skb, int l3proto, int l4proto, int srcport, int dstport)
{
	int l4src = 0, l4dst = 0;

	if (skb->protocol != htons(l3proto))
		return 0;
	if (l3proto == ETH_P_IP && ip_hdr(skb)->protocol != l4proto)
		return 0;
	if (l3proto == ETH_P_IPV6 && ipv6_hdr(skb)->nexthdr != l4proto)
		return 0 ;
	if (l4proto == IPPROTO_UDP) {
		l4src = udp_hdr(skb)->source;
		l4dst = udp_hdr(skb)->dest;
	} else if (l4proto == IPPROTO_TCP) {
		l4src = tcp_hdr(skb)->source;
		l4dst = tcp_hdr(skb)->dest;
	}
	if ((srcport && l4src != htons(srcport)) ||
	    (dstport && l4dst != htons(dstport)))
		return 0;

	return 1;
}
static inline int
cf_l3v4(struct sk_buff *skb, int l4proto, int srcport, int dstport)
{
	return cf_l3(skb, ETH_P_IP, l4proto, srcport, dstport);
}
static inline int
cf_l3v6(struct sk_buff *skb, int l4proto, int srcport, int dstport)
{
	return cf_l3(skb, ETH_P_IPV6, l4proto, srcport, dstport);
}
static inline int is_dhcp(struct sk_buff *skb)
{
	return cf_l3v4(skb, IPPROTO_UDP, 68, 67) ||
	       cf_l3v4(skb, IPPROTO_UDP, 67, 68);
}
static inline int is_dhcp6(struct sk_buff *skb)
{
	return cf_l3v6(skb, IPPROTO_UDP, 546, 547) ||
	       cf_l3v6(skb, IPPROTO_UDP, 547, 546);
}
static inline int is_dns(struct sk_buff *skb)
{
	return cf_l3v4(skb, IPPROTO_UDP, 0, 53) ||
	       cf_l3v6(skb, IPPROTO_UDP, 0, 53);
}
static inline int is_dns_reply(struct sk_buff *skb)
{
	return cf_l3v4(skb, IPPROTO_UDP, 53, 0) ||
	       cf_l3v6(skb, IPPROTO_UDP, 53, 0);
}
static inline int is_http(struct sk_buff *skb)
{
	return cf_l3v4(skb, IPPROTO_TCP, 0, 80) ||
	       cf_l3v6(skb, IPPROTO_TCP, 0, 80);
}
static inline int is_https(struct sk_buff *skb)
{
	return cf_l3v4(skb, IPPROTO_TCP, 0, 443) ||
	       cf_l3v6(skb, IPPROTO_TCP, 0, 443);
}

#endif /* _DPI_LOCAL_H */
