/*
 * <:copyright-BRCM:2020:DUAL/GPL:standard 
 * 
 *    Copyright (c) 2020 Broadcom 
 *    All Rights Reserved
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2, as published by
 * the Free Software Foundation (the "GPL").
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * 
 * A copy of the GPL is available at http://www.broadcom.com/licenses/GPLv2.php, or by
 * writing to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 * 
 * :>
 */

#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/skbuff.h>

#include "dpi_local.h"

/* --- types and consts --- */
#define DHCP_OPT_MAGIC		0x63825363

#define BOOTREQUEST		1
#define BOOTREPLY		2

/* DHCP operations */
#define DHCPDISCOVER		1
#define DHCPOFFER		2
#define DHCPREQUEST		3
#define DHCPDECLINE		4
#define DHCPACK			5
#define DHCPNAK			6
#define DHCPRELEASE		7
#define DHCPINFORM		8

/* DHCP option types */
#define DHCP_OPT_NETWORK_MASK	1
#define DHCP_OPT_ROUTERS	3
#define DHCP_OPT_SERVERS_DNS	6
#define DHCP_OPT_SERVERS_LOG	7
#define DHCP_OPT_HOSTNAME	12
#define DHCP_OPT_ADDR_REQUESTED	50
#define DHCP_OPT_ADDR_LEASETIME	51
#define DHCP_OPT_MESSAGE_TYPE	53
#define DHCP_OPT_SERVER_ID	54
#define DHCP_OPT_REQUEST_PARAMS	55
#define DHCP_OPT_MESSAGE	56
#define DHCP_OPT_MAX_MSG_SIZE	57
#define DHCP_OPT_T1_RENEW_TIME	58
#define DHCP_OPT_T2_REBIND_TIME	59
#define DHCP_OPT_VENDOR_CLASSID	60
#define DHCP_OPT_CLIENT_ID	61
#define DHCP_OPT_TFTP_SNAME	66
#define DHCP_OPT_TFTP_FILENAME	67
#define DHCP_OPT_RAPID_COMMIT	80
#define DHCP_OPT_RELAY_INFO	82
#define DHCP_OPT_SYSTEM_ARCH	93
#define DHCP_OPT_NET_DEV_IFACE	94
#define DHCP_OPT_SERVERS_TFTP	150
#define DHCP_OPT_END		255

struct dhcp_pkt {
	u8	op;
	u8	htype;
	u8	hlen;
	u8	hops;

	u32	xid;

	u16	secs;
	u16	flags;

	struct in_addr	ciaddr;
	struct in_addr	yiaddr;
	struct in_addr	siaddr;
	struct in_addr	giaddr;

	u8	chaddr[16];
	u8	sname[64];
	u8	file[128];
	u32	magic;		/* DHCP magic value */
	u8	options[0];	/* options */
} __packed;

/* --- functions --- */
static int ignore_dhcp_packet(struct dpi_classify_parms *p,
			      struct dhcp_pkt *dp, int total_len)
{
	if (total_len < sizeof(*dp)) {
		pr_debug("DHCP packet too small (%d, must be >= %lu)\n",
			 total_len, (unsigned long)sizeof(*dp));
		return 1;
	}

	if (is_zero_ether_addr(dp->chaddr)) {
		pr_debug("DHCP packet contains empty MAC\n");
		return 1;
	}

	return 0;
}

static void
dhcp_device_check(struct dpi_classify_parms *p, struct dhcp_pkt *dp)
{
	if (dp->op != BOOTREQUEST) {
		pr_debug("don't reclassify non-request\n");
		return;
	}

	/* if the source MAC is not the same as the requester, change the DPI
	 * classification to use the chaddr in the DHCP packet. */
	if (!p->dev || !ether_addr_equal(p->dev->mac, dp->chaddr)) {
		if (p->dev)
			atomic_dec(&p->dev->refcount);

		p->dev = dpi_dev_find_or_alloc(dp->chaddr);
		if (p->dev)
			atomic_inc(&p->dev->refcount);
	}
}

static void dhcp_ack(struct dpi_classify_parms *p, struct dhcp_pkt *dp)
{
	u32 *ciaddr = (u32*)&dp->ciaddr;
	u32 *yiaddr = (u32*)&dp->yiaddr;

	/* Allocate IP->MAC mapping for new DHCP issuances */
	pr_debug("DHCPACK %pI4 for %pM\n", yiaddr, dp->chaddr);

	/* if renewal changed addresses, free the old address */
	if (memcmp(ciaddr, yiaddr, sizeof(*ciaddr))) {
		struct dpi_ip *old = dpi_ip_find((u8*)ciaddr, NFPROTO_IPV4);
		if (old)
			dpi_ip_free(old);
	}

	dpi_ip_alloc((u8*)yiaddr, NFPROTO_IPV4, dp->chaddr);
}

void dpi_parse_dhcp(struct dpi_classify_parms *p)
{
	struct udphdr *uh	= udp_hdr(p->skb);
	struct dhcp_pkt *dp	= (void*)(uh + 1);
	int total_len		= ntohs(uh->len);
	u8 *data		= &dp->options[0];

	if (ignore_dhcp_packet(p, dp, total_len))
		return;

	dhcp_device_check(p, dp);
	if (!p->dev) {
		pr_debug("DHCP packet, but no DPI device\n");
		return;
	}

	/* copy the DHCP hostname if it exists */
	if (strnlen(dp->sname, sizeof(dp->sname))) {
		memset(p->dev->hostname, 0, sizeof(p->dev->hostname));
		memcpy(p->dev->hostname, dp->sname, sizeof(dp->sname));
		pr_debug("DHCP sname '%s'\n", p->dev->hostname);
	}

	if (dp->magic != htonl(DHCP_OPT_MAGIC)) {
		pr_debug("skipping BOOTP message\n");
		return;
	}

	total_len -= offsetof(struct dhcp_pkt, options);
	while (total_len > 0) {
		u8 tag = data[0];
		u8 len = total_len > 1 ? data[1] : 0;

		pr_debug("tag %d, len %d\n", tag, len);

		/* validate */
		if (len > total_len || !len)
			break;

		/* copy hostname from DHCP options */
		if (tag == DHCP_OPT_HOSTNAME) {
			size_t strlen = len;
			strlen = min(strlen, sizeof(p->dev->hostname) - 1);
			memset(p->dev->hostname, 0, sizeof(p->dev->hostname));
			memcpy(p->dev->hostname, &data[2], strlen);
			pr_debug("DHCP hostname '%s'\n", p->dev->hostname);
			break;
		}
		if (tag == DHCP_OPT_MESSAGE_TYPE) {
			if (data[2] == DHCPACK)
				dhcp_ack(p, dp);
		}

		total_len	-= len + 2;
		data		+= len + 2;
	}
}
