/*
 * <:copyright-BRCM:2017:DUAL/GPL:standard 
 * 
 *    Copyright (c) 2017 Broadcom 
 *    All Rights Reserved
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2, as published by
 * the Free Software Foundation (the "GPL").
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * 
 * A copy of the GPL is available at http://www.broadcom.com/licenses/GPLv2.php, or by
 * writing to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 * 
 * :>
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/hashtable.h>
#include <linux/list.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/etherdevice.h>
#include <net/netfilter/nf_conntrack.h>
#include <net/netfilter/nf_conntrack_acct.h>
#include <linux/workqueue.h>

#include <linux/dpi.h>

#include "dpi_local.h"

#define APP_TABLE_BITS		6
#define DEV_TABLE_BITS		4
#define APPINST_TABLE_BITS	6
#define URL_TABLE_BITS		6
#define IP_TABLE_BITS		3
#define URL_MAX_HASH_CHARS	20

#define APP_TABLE_MAX_SIZE	10000
#define DEV_TABLE_MAX_SIZE	1024
#define APPINST_TABLE_MAX_SIZE	1000000
#define URL_TABLE_MAX_SIZE	10000
#define IP_TABLE_MAX_SIZE	1024

/* ----- variables ----- */
#define DEFINE_DPI_TABLE(name, bits) \
	struct name { \
		DECLARE_HASHTABLE(list, bits); \
		struct kmem_cache *cache; \
		unsigned int size; \
		unsigned int max_size; \
		unsigned int clear_threshold; \
		spinlock_t lock; \
	} name
DEFINE_DPI_TABLE(app_table, APP_TABLE_BITS);
DEFINE_DPI_TABLE(dev_table, DEV_TABLE_BITS);
DEFINE_DPI_TABLE(appinst_table, APPINST_TABLE_BITS);
DEFINE_DPI_TABLE(url_table, URL_TABLE_BITS);
DEFINE_DPI_TABLE(ip_table, IP_TABLE_BITS);

static void dpi_clear_table(struct work_struct *w);
static DECLARE_WORK(dpi_tables_work, dpi_clear_table);
static unsigned long dpi_tables_work_flags;

static void dpi_reset_tables(unsigned long flags, int reset_stats)
{
	struct dpi_appinst *appinst;
	struct dpi_app *app;
	struct dpi_dev *dev;
	struct dpi_url *url;
	struct hlist_node *tmp;
	int i;

	/*
	 * Delete all entries with 0 refcount. For active entries (with nonzero
	 * reference count), wipe the current net stats if reset_stats.
	 */

	if (!flags || test_bit(DPI_RESET_APPINST_BIT, &flags)) {
		spin_lock_bh(&appinst_table.lock);
		hash_for_each_safe(appinst_table.list, i, tmp, appinst, node) {
			if (atomic_read(&appinst->refcount)) {
				if (reset_stats) {
					memset(&appinst->us, 0, sizeof(appinst->us));
					memset(&appinst->ds, 0, sizeof(appinst->ds));
				}
				continue;
			}

			if (appinst->app)
				atomic_dec(&appinst->app->refcount);
			if (appinst->dev)
				atomic_dec(&appinst->dev->refcount);

			hash_del(&appinst->node);
			kmem_cache_free(appinst_table.cache, appinst);
			appinst_table.size--;
		}
		spin_unlock_bh(&appinst_table.lock);
	}

	if (!flags || test_bit(DPI_RESET_APP_BIT, &flags)) {
		spin_lock_bh(&app_table.lock);
		hash_for_each_safe(app_table.list, i, tmp, app, node) {
			if (atomic_read(&app->refcount))
				continue;

			hash_del(&app->node);
			kmem_cache_free(app_table.cache, app);
			app_table.size--;
		}
		spin_unlock_bh(&app_table.lock);
	}

	if (!flags || test_bit(DPI_RESET_DEV_BIT, &flags)) {
		spin_lock_bh(&dev_table.lock);
		hash_for_each_safe(dev_table.list, i, tmp, dev, node) {
			if (atomic_read(&dev->refcount)) {
				if (reset_stats) {
					memset(&dev->us, 0, sizeof(dev->us));
					memset(&dev->ds, 0, sizeof(dev->ds));
				}
				continue;
			}

			hash_del(&dev->node);
			kmem_cache_free(dev_table.cache, dev);
			dev_table.size--;
		}
		spin_unlock_bh(&dev_table.lock);
	}

	if (!flags || test_bit(DPI_RESET_URL_BIT, &flags)) {
		spin_lock_bh(&url_table.lock);
		hash_for_each_safe(url_table.list, i, tmp, url, node) {
			if (atomic_read(&url->refcount))
				continue;

			hash_del(&url->node);
			kmem_cache_free(url_table.cache, url);
			url_table.size--;
		}
		spin_unlock_bh(&url_table.lock);
	}
}

static void dpi_clear_table(struct work_struct *w)
{
	dpi_reset_tables(dpi_tables_work_flags, 0);
	dpi_tables_work_flags = 0;
}

/* ----- public functions ----- */
struct dpi_dev *dpi_dev_find_or_alloc(u8 *mac)
{
	struct hlist_head *h;
	struct dpi_dev *entry;
	u32 val = *((u32 *) &mac[0]) ^ *((u16 *) &mac[4]);

	if (is_zero_ether_addr(mac))
		return NULL;

	spin_lock_bh(&dev_table.lock);

	/* search for existing entry */
	h = &dev_table.list[hash_min(val, DEV_TABLE_BITS)];
	hlist_for_each_entry(entry, h, node) {
		if (ether_addr_equal(entry->mac, mac))
			goto out;
	}

	if (dev_table.size > dev_table.clear_threshold) {
		set_bit(DPI_RESET_APPINST_BIT, &dpi_tables_work_flags);
		set_bit(DPI_RESET_DEV_BIT, &dpi_tables_work_flags);
		schedule_work(&dpi_tables_work);
		pr_info("dev table almost full (%u out of %u), "
			"scheduled work to remove inactive entries\n",
			dev_table.size, dev_table.max_size);
		if (dev_table.size >= dev_table.max_size) {
			pr_err("dev table full (%u entries)\n", dev_table.size);
			entry = NULL;
			goto out;
		}
	}
	/* allocate new entry if not found */
	entry = kmem_cache_zalloc(dev_table.cache, GFP_ATOMIC);
	if (!entry) {
		pr_err("unable to allocate entry\n");
		goto out;
	}
	memcpy(entry->mac, mac, sizeof(entry->mac));
	entry->prio = (u16)-1;

	/* add entry to hlist */
	pr_debug("new dev %pM\n", entry->mac);
	INIT_HLIST_NODE(&entry->node);
	hlist_add_head(&entry->node, h);
	dev_table.size++;
	dpi_stats.dev_count++;

out:
	spin_unlock_bh(&dev_table.lock);
	return entry;
}

struct dpi_app *dpi_app_find_or_alloc(u32 app_id)
{
	struct hlist_head *h;
	struct dpi_app *entry;

	spin_lock_bh(&app_table.lock);

	/* search for existing entry */
	h = &app_table.list[hash_min(app_id, APP_TABLE_BITS)];
	hlist_for_each_entry(entry, h, node) {
		if (entry->app_id == app_id)
			goto out;
	}

	if (app_table.size > app_table.clear_threshold) {
		set_bit(DPI_RESET_APPINST_BIT, &dpi_tables_work_flags);
		set_bit(DPI_RESET_APP_BIT, &dpi_tables_work_flags);
		schedule_work(&dpi_tables_work);
		pr_info("app table almost full (%u out of %u), "
			"scheduled work to remove inactive entries\n",
			app_table.size, app_table.max_size);
		if (app_table.size >= app_table.max_size) {
			pr_err("app table full (%u entries)\n", app_table.size);
			entry = NULL;
			goto out;
		}
	}
	/* allocate new entry if not found */
	entry = kmem_cache_zalloc(app_table.cache, GFP_ATOMIC);
	if (!entry) {
		pr_err("unable to allocate entry\n");
		goto out;
	}
	entry->app_id = app_id;

	/* add entry to hlist */
	pr_debug("new app %08x\n", entry->app_id);
	INIT_HLIST_NODE(&entry->node);
	hlist_add_head(&entry->node, h);
	app_table.size++;
	dpi_stats.app_count++;

out:
	spin_unlock_bh(&app_table.lock);
	return entry;
}

struct dpi_appinst *dpi_appinst_find_or_alloc(struct dpi_app *app,
					      struct dpi_dev *dev)
{
	struct hlist_head *h;
	struct dpi_appinst *entry;
	u32 val = (app ? app->app_id : 0) ^
		  *((u32 *) &dev->mac[0]) ^
		  *((u16 *) &dev->mac[4]);

	spin_lock_bh(&appinst_table.lock);

	/* search for existing entry */
	h = &appinst_table.list[hash_min(val, APPINST_TABLE_BITS)];
	hlist_for_each_entry(entry, h, node) {
		if (entry->app == app && entry->dev == dev)
			goto out;
	}

	if (appinst_table.size > appinst_table.clear_threshold) {
		set_bit(DPI_RESET_APPINST_BIT, &dpi_tables_work_flags);
		schedule_work(&dpi_tables_work);
		pr_info("appinst table almost full (%u out of %u), "
			"scheduled work to remove inactive entries\n",
			appinst_table.size, appinst_table.max_size);
		if (appinst_table.size >= appinst_table.max_size) {
			pr_info("appinst table full (%u entries)\n",
				appinst_table.size);
			entry = NULL;
			goto out;
		}
	}
	/* allocate new entry if not found */
	entry = kmem_cache_zalloc(appinst_table.cache, GFP_ATOMIC);
	if (!entry) {
		pr_err("unable to allocate entry\n");
		goto out;
	}
	entry->app = app;
	if (app)
		atomic_inc(&app->refcount);
	entry->dev = dev;
	if (dev)
		atomic_inc(&dev->refcount);

	/* add entry to hlist */
	pr_debug("new appinst [%08x %pM]\n",
		 app ? app->app_id : 0, dev->mac);
	INIT_HLIST_NODE(&entry->node);
	hlist_add_head(&entry->node, h);
	appinst_table.size++;
	dpi_stats.appinst_count++;

out:
	spin_unlock_bh(&appinst_table.lock);
	return entry;
}

static u32 __hash_string(char *string, int len, int bits)
{
	u32 hash = 0x61C88647;
	int i;

	for (i = 0; i < URL_MAX_HASH_CHARS && *string; i++, string++)
		hash = hash * (*string);

	return hash >> (32 - bits);
}

struct dpi_url *dpi_url_find_or_alloc(char *hostname, int len)
{
	struct hlist_head *h;
	struct dpi_url *entry;
	u32 hash = __hash_string(hostname, len, URL_TABLE_BITS);

	spin_lock_bh(&url_table.lock);

	/* search for existing entry */
	h = &url_table.list[hash];
	hlist_for_each_entry(entry, h, node) {
		if (memcmp(entry->hostname, hostname, entry->len) == 0)
			goto out;
	}

	if (url_table.size > url_table.clear_threshold) {
		set_bit(DPI_RESET_URL_BIT, &dpi_tables_work_flags);
		schedule_work(&dpi_tables_work);
		pr_info("url table almost full (%u out of %u), "
			"scheduled work to remove inactive entries\n",
			url_table.size, url_table.max_size);
		if (url_table.size >= url_table.max_size) {
			pr_err("url table full (%u entries)\n", url_table.size);
			entry = NULL;
			goto out;
		}
	}
	/* allocate new entry if not found */
	entry = kmem_cache_zalloc(url_table.cache, GFP_ATOMIC);
	if (!entry) {
		pr_err("unable to allocate entry\n");
		goto out;
	}

	entry->len = min(len, DPI_URLINFO_MAX_HOST_LEN - 1);
	if (entry->len != len)
		pr_debug("truncating long URL %s\n", hostname);
	/* copy string and explicitly add null terminator */
	memcpy(entry->hostname, hostname, entry->len);
	entry->hostname[entry->len] = '\0';

	/* add entry to hlist */
	pr_debug("new url %s\n", entry->hostname);
	INIT_HLIST_NODE(&entry->node);
	hlist_add_head(&entry->node, h);
	url_table.size++;
	dpi_stats.url_count++;

out:
	spin_unlock_bh(&url_table.lock);
	return entry;
}

static inline u32 l3hash(u32 *ip, int l3proto)
{
	u32 val = 0;

	if (l3proto == NFPROTO_IPV4)
		val = hash_min(*ip, IP_TABLE_BITS);
	if (l3proto == NFPROTO_IPV6) {
		val = hash_32(ip[0], 32) + hash_32(ip[1], 32) +
		      hash_32(ip[2], 32) + hash_32(ip[3], 32);
		val = hash_min(val, IP_TABLE_BITS);
	}

	return val;
}

static inline int l3addr_length(int l3proto)
{
	if (l3proto == NFPROTO_IPV4)
		return FIELD_SIZEOF(struct iphdr, saddr);
	if (l3proto == NFPROTO_IPV6)
		return FIELD_SIZEOF(struct ipv6hdr, saddr);
	return 0;
}

static struct dpi_ip *__dpi_ip_find(u8 *ip, int l3proto)
{
	struct dpi_ip *entry;
	int len;
	u32 val;

	if (!ip || !l3addr_length(l3proto))
		return NULL;

	val = l3hash((u32*)ip, l3proto);
	len = l3addr_length(l3proto);

	hash_for_each_possible(ip_table.list, entry, node, val) {
		if (entry->l3proto != l3proto)
			continue;
		if (!memcmp(&entry->ip, ip, len))
			break;
	}

	return entry;
}

struct dpi_ip *dpi_ip_find(u8 *ip, int l3proto)
{
	struct dpi_ip *entry;

	spin_lock_bh(&ip_table.lock);
	entry = __dpi_ip_find(ip, l3proto);
	spin_unlock_bh(&ip_table.lock);

	return entry;
}

struct dpi_ip *dpi_ip_find_by_skb(struct sk_buff *skb)
{
	void *ip = NULL;
	int l3proto;

	if (skb->protocol == htons(ETH_P_IP)) {
		l3proto	= NFPROTO_IPV4;
		ip	= is_netdev_wan(skb->dev) ?
				&ip_hdr(skb)->daddr :
				&ip_hdr(skb)->saddr;
	} else if (skb->protocol == htons(ETH_P_IPV6)) {
		l3proto	= NFPROTO_IPV6;
		ip	= is_netdev_wan(skb->dev) ?
				&ipv6_hdr(skb)->daddr :
				&ipv6_hdr(skb)->saddr;
	}

	return dpi_ip_find(ip, l3proto);
}

static void __dpi_ip_free(struct dpi_ip *entry)
{
	hash_del(&entry->node);
	kmem_cache_free(ip_table.cache, entry);
	ip_table.size--;
}

struct dpi_ip *dpi_ip_alloc(u8 *ip, int l3proto, u8* mac)
{
	struct hlist_head *h;
	struct dpi_ip *entry;
	struct dpi_ip *new_entry = NULL;
	struct hlist_node *n;
	int bkt;

	if (!ip || !l3addr_length(l3proto))
		return NULL;

	spin_lock_bh(&ip_table.lock);
	entry = __dpi_ip_find(ip, l3proto);
	if (entry) {
		memcpy(entry->mac, mac, sizeof(entry->mac));
		goto check_duplicate_macs;
	}
	entry = kmem_cache_zalloc(ip_table.cache, GFP_ATOMIC);
	if (!entry) {
		pr_err("enable to allocate entry\n");
		goto out;
	}
	entry->l3proto = l3proto;
	memcpy(&entry->ip, ip, l3addr_length(l3proto));
	memcpy(entry->mac, mac, sizeof(entry->mac));

	/* add entry to hlist */
	if (l3proto == NFPROTO_IPV4)
		pr_debug("new ip %pI4 -> %pM\n", ip, mac);
	else if (l3proto == NFPROTO_IPV6)
		pr_debug("new ip %pI6c -> %pM\n", ip, mac);
	INIT_HLIST_NODE(&entry->node);
	h = &ip_table.list[l3hash((u32*)ip, l3proto)];
	hlist_add_head(&entry->node, h);
	ip_table.size++;
	new_entry = entry;

check_duplicate_macs:
	/* Remove other entries that match the same mac as the new entry. This
	 * covers cases where a LAN client changes IP addresses. */
	hash_for_each_safe(ip_table.list, bkt, n, entry, node) {
		if (entry == new_entry)
			continue;
		if (ether_addr_equal(entry->mac, mac))
			__dpi_ip_free(entry);
	}

out:
	spin_unlock_bh(&ip_table.lock);
	return entry;
}

void dpi_ip_free(struct dpi_ip *entry)
{
	spin_lock_bh(&ip_table.lock);
	__dpi_ip_free(entry);
	spin_unlock_bh(&ip_table.lock);
}

static void dpi_ct_save_stats(struct nf_conn *ct, int dir)
{
	struct dpi_ct_stats *appinst_stats;
	struct dpi_ct_stats *dev_stats;
	struct nf_conn_acct *acct;
	struct nf_conn_counter *counter;
	uint64_t packets, bytes;
#if IS_ENABLED(CONFIG_BLOG)
	BlogFcStats_t fast_stats = {0};
#endif

	acct = nf_conn_acct_find(ct);
	if (!acct)
		return;

	counter	= acct->counter;
	packets	= atomic64_read(&counter[dir].packets);
	bytes	= atomic64_read(&counter[dir].bytes);

#if IS_ENABLED(CONFIG_BLOG)
	blog_ct_get_stats(ct, ct->bcm_ext.blog_key[dir], dir, &fast_stats);
	packets	+= fast_stats.rx_packets;
	bytes	+= fast_stats.rx_bytes;
#endif

	pr_debug("app:%px dev:%px appinst:%px dir:%s counter:[%llu, %llu]",
		 ct->bcm_ext.dpi.app, ct->bcm_ext.dpi.dev,
		 ct->bcm_ext.dpi.appinst,
		 dir == IP_CT_DIR_ORIGINAL ? "orig" : "reply",
		 packets, bytes);

	/* update cumulative appinst stats */
	if (ct->bcm_ext.dpi.appinst) {
		appinst_stats = dpi_appinst_stats(ct, dir);

		appinst_stats->pkts += packets;
		appinst_stats->bytes += bytes;

		pr_debug(" appinst_stats:[%llu, %llu]",
			 appinst_stats->pkts,
			 appinst_stats->bytes);
	}

	/* update cumulative device stats */
	if (ct->bcm_ext.dpi.dev) {
		dev_stats = dpi_dev_stats(ct, dir);

		dev_stats->pkts += packets;
		dev_stats->bytes += bytes;

		pr_debug(" dev_stats:[%llu, %llu]",
			 dev_stats->pkts,
			 dev_stats->bytes);
	}

	pr_debug("\n");
}

void dpi_ct_destroy(struct nf_conn *ct)
{
	dpi_ct_save_stats(ct, IP_CT_DIR_ORIGINAL);
	if ((test_bit(IPS_SEEN_REPLY_BIT, &ct->status)))
		dpi_ct_save_stats(ct, IP_CT_DIR_REPLY);

	if (ct->bcm_ext.dpi.appinst)
		atomic_dec(&ct->bcm_ext.dpi.appinst->refcount);
	if (ct->bcm_ext.dpi.dev)
		atomic_dec(&ct->bcm_ext.dpi.dev->refcount);
	if (ct->bcm_ext.dpi.app)
		atomic_dec(&ct->bcm_ext.dpi.app->refcount);
	if (ct->bcm_ext.dpi.url)
		atomic_dec(&ct->bcm_ext.dpi.url->refcount);
}

void dpi_reset_stats(unsigned long flags)
{
	dpi_reset_tables(flags, 1);
}

/* ----- local functions ----- */
#define DEFINE_DPI_HLIST_SEQ(name, table, entry_type, fmt, show_fun) \
	static void *dpi_##name##_seq_start(struct seq_file *s, loff_t *pos) \
	{ \
		loff_t *spos = s->private; \
		*spos = *pos - 1; \
		if (*pos == 0) \
			return SEQ_START_TOKEN; \
		if (*spos >= HASH_SIZE(table.list)) \
			return NULL; \
		return spos; \
	} \
	static void *dpi_##name##_seq_next(struct seq_file *s, void *v, \
					    loff_t *pos) \
	{ \
		loff_t *spos = s->private; \
		(*pos)++; \
		(*spos)++; \
		if (*spos >= HASH_SIZE(table.list)) \
			return NULL; \
		return spos; \
	} \
	static void dpi_##name##_seq_stop(struct seq_file *s, void *v) \
	{ \
	} \
	static int dpi_##name##_seq_show(struct seq_file *s, void *v) \
	{ \
		loff_t *spos = s->private; \
		struct hlist_head *h = &table.list[*spos]; \
		entry_type *entry; \
		\
		if (v == SEQ_START_TOKEN) { \
			seq_printf(s, fmt "\n"); \
			return 0; \
		} \
		\
		/* print each entry in the bucket */ \
		hlist_for_each_entry(entry, h, node) \
			if (!show_fun(s, entry)) \
				seq_printf(s, "\n"); \
		return 0; \
	} \
	static const struct seq_operations dpi_##name##_seq_ops = { \
		.start	= dpi_##name##_seq_start, \
		.next	= dpi_##name##_seq_next, \
		.stop	= dpi_##name##_seq_stop, \
		.show	= dpi_##name##_seq_show \
	}; \
	static int dpi_##name##_open(struct inode *inode, struct file *file) \
	{ \
		return seq_open_private(file, &dpi_##name##_seq_ops, \
					sizeof(loff_t)); \
	} \
	static const struct file_operations dpi_##name##_fops = { \
		.open		= dpi_##name##_open, \
		.read		= seq_read, \
		.llseek		= seq_lseek, \
		.release	= seq_release_private, \
	}

#define dpi_app_desc	"app_id"
#define dpi_dev_desc	"mac vendor os os_class category family dev \"hostname\""
#define dpi_stats_desc	"up_pkt up_byte down_pkt down_byte"

static int dpi_app_entry_show(struct seq_file *s, struct dpi_app *entry)
{
	/* don't print behaviour in app_id */
	seq_printf(s, "%u", entry ? entry->app_id : 0);
	return 0;
}
DEFINE_DPI_HLIST_SEQ(app, app_table, struct dpi_app, dpi_app_desc,
		     dpi_app_entry_show);

static int dpi_dev_entry_show(struct seq_file *s, struct dpi_dev *entry)
{
	seq_printf(s, "%pM %u %u %u %u %u %u \"%s\"",
		   entry->mac,
		   entry->vendor,
		   entry->os,
		   entry->os_class,
		   entry->category,
		   entry->family,
		   entry->dev_id,
		   strlen(entry->hostname) ? entry->hostname : NULL);
	return 0;
}
static int dpi_dev_entry_stats_show(struct seq_file *s, struct dpi_dev *entry)
{
	dpi_dev_entry_show(s, entry);

	seq_printf(s, " %llu %llu %llu %llu",
		   entry->us.pkts,
		   entry->us.bytes,
		   entry->ds.pkts,
		   entry->ds.bytes);
	return 0;
}
DEFINE_DPI_HLIST_SEQ(dev, dev_table, struct dpi_dev,
		     dpi_dev_desc " " dpi_stats_desc,
		     dpi_dev_entry_stats_show);

static int dpi_appinst_entry_show(struct seq_file *s,
				  struct dpi_appinst *entry)
{
	BUG_ON(!entry->dev);

	dpi_app_entry_show(s, entry->app);
	seq_puts(s, " ");
	dpi_dev_entry_show(s, entry->dev);

	seq_printf(s, " %llu %llu %llu %llu",
		   entry->us.pkts,
		   entry->us.bytes,
		   entry->ds.pkts,
		   entry->ds.bytes);
	return 0;
}
DEFINE_DPI_HLIST_SEQ(appinst, appinst_table, struct dpi_appinst,
		     dpi_app_desc " " dpi_dev_desc " " dpi_stats_desc,
		     dpi_appinst_entry_show);


static int dpi_url_entry_show(struct seq_file *s, struct dpi_url *entry)
{
	seq_printf(s, "%s", entry->hostname);
	return 0;
}
DEFINE_DPI_HLIST_SEQ(url, url_table, struct dpi_url, "host_name",
		     dpi_url_entry_show);

static int dpi_ip_entry_show(struct seq_file *s, struct dpi_ip *entry)
{
	if (entry->l3proto == NFPROTO_IPV4)
		seq_printf(s, "4 %pI4 %pM", &entry->ip, entry->mac);
	if (entry->l3proto == NFPROTO_IPV6)
		seq_printf(s, "6 %pI6c %pM", &entry->ip6, entry->mac);
	return 0;
}
DEFINE_DPI_HLIST_SEQ(ip, ip_table, struct dpi_ip, "ipver ip mac",
		     dpi_ip_entry_show);

#define init_table(name, struct_name, max_entries) \
	({ \
		int __ret = 0; \
		spin_lock_init(&name.lock); \
		hash_init(name.list); \
		name.max_size = max_entries; \
		name.clear_threshold = max_entries / 100 * 98; \
		name.cache = KMEM_CACHE(struct_name, SLAB_HWCACHE_ALIGN); \
		if (!name.cache) { \
			pr_err("couldn't allocate "#name"\n"); \
			__ret = -1; \
		} \
		__ret; \
	})

#define init_proc_entry(name, permissions, parent, fops) \
	({ \
		struct proc_dir_entry *__pde; \
		int __ret = 0; \
		__pde = proc_create(name, permissions, parent, &fops); \
		if (!__pde) { \
			pr_err("couldn't create proc entry '"#name"'\n"); \
			__ret = -1; \
		} \
		__ret; \
	})

int __init dpi_init_tables(void)
{
	if (init_table(app_table, dpi_app, APP_TABLE_MAX_SIZE))
		goto err;
	if (init_table(dev_table, dpi_dev, DEV_TABLE_MAX_SIZE))
		goto err_free_app_table;
	if (init_table(appinst_table, dpi_appinst, APPINST_TABLE_MAX_SIZE))
		goto err_free_dev_table;
	if (init_table(url_table, dpi_url, URL_TABLE_MAX_SIZE))
		goto err_free_appinst_table;
	if (init_table(ip_table, dpi_ip, IP_TABLE_MAX_SIZE))
		goto err_free_url_table;

	/* create proc entries */
	if (init_proc_entry("appinsts", 0440, dpi_dir, dpi_appinst_fops))
		goto err_free_ip_table;
	if (init_proc_entry("apps", 0440, dpi_dir, dpi_app_fops))
		goto err_free_appinsts;
	if (init_proc_entry("devices", 0440, dpi_dir, dpi_dev_fops))
		goto err_free_apps;
	if (init_proc_entry("urls", 0440, dpi_dir, dpi_url_fops))
		goto err_free_devices;
	if (init_proc_entry("ips", 0440, dpi_dir, dpi_ip_fops))
		goto err_free_urls;

	return 0;

err_free_urls:
	remove_proc_entry("urls", dpi_dir);
err_free_devices:
	remove_proc_entry("devices", dpi_dir);
err_free_apps:
	remove_proc_entry("apps", dpi_dir);
err_free_appinsts:
	remove_proc_entry("appinsts", dpi_dir);
err_free_ip_table:
	kmem_cache_destroy(ip_table.cache);
err_free_url_table:
	kmem_cache_destroy(url_table.cache);
err_free_appinst_table:
	kmem_cache_destroy(appinst_table.cache);
err_free_dev_table:
	kmem_cache_destroy(dev_table.cache);
err_free_app_table:
	kmem_cache_destroy(app_table.cache);
err:
	return -ENOMEM;
}

#define cleanup_dpi_table(name, struct_name) \
	do { \
		struct hlist_node *__tmp; \
		struct struct_name *__item; \
		int __i; \
		\
		spin_lock_bh(&name.lock); \
		hash_for_each_safe(name.list, __i, __tmp, __item, node) \
			kmem_cache_free(name.cache, __item); \
		kmem_cache_destroy(name.cache); \
		spin_unlock_bh(&name.lock); \
	} while (0)
void __exit dpi_deinit_tables(void)
{
	remove_proc_entry("devices", dpi_dir);
	remove_proc_entry("apps", dpi_dir);
	remove_proc_entry("appinsts", dpi_dir);
	remove_proc_entry("urls", dpi_dir);
	remove_proc_entry("ips", dpi_dir);
	proc_remove(dpi_dir);

	cancel_work_sync(&dpi_tables_work);

	cleanup_dpi_table(app_table, dpi_app);
	cleanup_dpi_table(dev_table, dpi_dev);
	cleanup_dpi_table(appinst_table, dpi_appinst);
	cleanup_dpi_table(url_table, dpi_url);
	cleanup_dpi_table(ip_table, dpi_ip);
}
