/*
 * <:copyright-BRCM:2019:DUAL/GPL:standard 
 * 
 *    Copyright (c) 2019 Broadcom 
 *    All Rights Reserved
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2, as published by
 * the Free Software Foundation (the "GPL").
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * 
 * A copy of the GPL is available at http://www.broadcom.com/licenses/GPLv2.php, or by
 * writing to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 * 
 * :>
 */

#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/skbuff.h>
#include <linux/ip.h>
#include <linux/ipv6.h>
#include <linux/proc_fs.h>
#include <net/netfilter/nf_conntrack.h>

#include <linux/dpi.h>
#include <bcmdpi.h>
#include "dpi_local.h"

struct test_map {
	int	src;
	int	dst;
	struct {
		u8	cat;
		u16	app;
		u8	beh;
	} app_id;
};
/* ----- variables ------ */
static unsigned int test_mode;
static struct test_map test_map[] = {
	{ .dst = 5000, .app_id = { 0,   2, 0} }, /* Yahoo Messenger */
	{ .dst = 5001, .app_id = { 1,   2, 0} }, /* Bittorrent */
	{ .dst = 5002, .app_id = { 3,   1, 0} }, /* FTP */
	{ .dst = 5003, .app_id = { 3,  84, 0} }, /* Dropbox */
	{ .dst = 5004, .app_id = { 4,  19, 0} }, /* Twitch.tv */
	{ .dst = 5005, .app_id = { 4, 112, 0} }, /* Youtube */
	{ .dst = 5006, .app_id = { 4, 132, 0} }, /* Netflix */
	{ .dst = 5007, .app_id = { 4, 193, 0} }, /* Amazon Video */
	{ .dst = 5008, .app_id = { 5,  95, 0} }, /* Gmail */
	{ .dst = 5009, .app_id = { 6,   1, 0} }, /* Skype */
	{ .dst = 5010, .app_id = { 6,   4, 0} }, /* Facetime */
	{ .dst = 5011, .app_id = {11,  55, 0} }, /* OpenVPN */
	{ .dst = 5012, .app_id = {13,   7, 0} }, /* Speedtest.net */
	{ .dst = 5013, .app_id = {14,   9, 0} }, /* Windows Update */
	{ .dst = 5014, .app_id = {19,   1, 0} }, /* PPTP */
	{ .dst = 5015, .app_id = { 8, 100, 0} }, /* Pokemon Netbattle */
	{ .dst = 5016, .app_id = { 8, 101, 0} }, /* RunUO-Ultima */
	{ .dst = 5017, .app_id = { 8, 102, 0} }, /* Soldat Dedicated */
	{ .dst = 5018, .app_id = { 8, 103, 0} }, /* Blizzard Entertainment */
	{ .dst = 5019, .app_id = { 8, 104, 0} }, /* RIFT */
	{ .dst = 5020, .app_id = { 8, 105, 0} }, /* TetriNET */
	{ .dst = 5021, .app_id = { 8, 106, 0} }, /* Tibia */
	{ .dst = 5022, .app_id = { 8, 107, 0} }, /* PlanetSide */
	{ .dst = 5023, .app_id = { 8, 108, 0} }, /* TripleA */
	{ .dst = 5024, .app_id = { 8, 110, 0} }, /* Unreal */
	{ .dst = 5025, .app_id = { 8, 111, 0} }, /* Valve Steam */
	{ .dst = 5026, .app_id = { 8, 112, 0} }, /* WesNOth */
	{ .dst = 5027, .app_id = { 8, 113, 0} }, /* Xpilot */
	{ .dst = 5028, .app_id = { 8, 114, 0} }, /* Swtor */
	{ .dst = 5029, .app_id = { 8, 115, 0} }, /* EVEOnline */
	{ .dst = 5030, .app_id = { 8, 116, 0} }, /* Hearthstone */
	{ .dst = 5031, .app_id = { 8, 117, 0} }, /* Guild Wars */
	{ .dst = 5032, .app_id = { 8, 118, 0} }, /* Zhong Hua Hero */
	{ .dst = 5033, .app_id = { 8, 119, 0} }, /* Wizard101 */
	{ .dst = 5034, .app_id = { 8, 120, 0} }, /* SD Gundam */
	{ .dst = 5035, .app_id = { 8, 121, 0} }, /* Prius */
	{ .dst = 5036, .app_id = { 8, 122, 0} }, /* Age of Conan */
	{ .dst = 5037, .app_id = { 8, 123, 0} }, /* RF Returns */
	{ .dst = 5038, .app_id = { 8, 124, 0} }, /* AION */
	{ .dst = 5039, .app_id = { 8, 125, 0} }, /* POPO Game */
	{ .dst = 5040, .app_id = { 8, 126, 0} }, /* War-Rock */
	{ .dst = 5041, .app_id = { 8, 127, 0} }, /* TEN Game */
	{ .dst = 5042, .app_id = { 8, 128, 0} }, /* LUNA2 */
	{ .dst = 5043, .app_id = { 8, 129, 0} }, /* Karos */
	{ .dst = 5044, .app_id = { 8, 130, 0} }, /* SPOnline */
	{ .dst = 5045, .app_id = { 8, 131, 0} }, /* RO Game */
	{ .dst = 5046, .app_id = { 8, 132, 0} }, /* StarCraft2 */
	{ .dst = 5047, .app_id = { 8, 133, 0} }, /* Itaiwanmj */
	{ .dst = 5048, .app_id = { 8, 134, 0} }, /* CMWEBGAME Game */
	{ .dst = 5049, .app_id = { 8, 135, 0} }, /* Beanfun Game */
	{ .dst = 5050, .app_id = { 8, 136, 0} }, /* JXW */
	{ .dst = 5051, .app_id = { 8, 137, 0} }, /* Nobol */
	{ .dst = 5052, .app_id = { 8, 138, 0} }, /* DragonNest */
	{ .dst = 5053, .app_id = { 8, 139, 0} }, /* BBonline */
	{ .dst = 5054, .app_id = { 8, 140, 0} }, /* Hangame */
	{ .dst = 5055, .app_id = { 8, 141, 0} }, /* Homygame */
	{ .dst = 5056, .app_id = { 8, 142, 0} }, /* Sony PlayStation */
	{ .dst = 5057, .app_id = { 8, 143, 0} }, /* Garena */
	{ .dst = 5058, .app_id = { 8, 144, 0} }, /* 91555 */
	{ .dst = 5059, .app_id = { 8, 145, 0} }, /* JJ Game */
	{ .dst = 5060, .app_id = { 8, 146, 0} }, /* YHgame */
	{ .dst = 5061, .app_id = { 8, 147, 0} }, /* Mdm365 */
	{ .dst = 5062, .app_id = { 8, 148, 0} }, /* 7fgame */
	{ .dst = 5063, .app_id = { 8, 149, 0} }, /* Dokee */
	{ .dst = 5064, .app_id = { 8, 150, 0} }, /* VSA */
	{ .dst = 5065, .app_id = { 8, 151, 0} }, /* Funtown */
	{ .dst = 5066, .app_id = { 8, 152, 0} }, /* SF Game */
	{ .dst = 5067, .app_id = { 8, 153, 0} }, /* 173kh */
	{ .dst = 5068, .app_id = { 8, 154, 0} }, /* Boyaapoker */
	{ .dst = 5069, .app_id = { 8, 155, 0} }, /* GameCenter */
	{ .dst = 5070, .app_id = { 8, 156, 0} }, /* Minecraft */
	{ .dst = 5071, .app_id = { 8, 157, 0} }, /* Dark Souls */
	{ .dst = 5072, .app_id = { 8, 158, 0} }, /* The Secret World */
	{ .dst = 5073, .app_id = { 8, 159, 0} }, /* World2 */
	{ .dst = 5074, .app_id = { 8, 160, 0} }, /* CrossFire */
	{ .dst = 5075, .app_id = { 8, 161, 0} }, /* XYQ */
	{ .dst = 5076, .app_id = { 8, 162, 0} }, /* Nexon */
	{ .dst = 5077, .app_id = { 8, 163, 0} }, /* Vindictus */
	{ .dst = 5078, .app_id = { 8, 164, 0} }, /* DotA */
	{ .dst = 5079, .app_id = { 8, 165, 0} }, /* PAYDAY */
	{ .dst = 5080, .app_id = { 8, 166, 0} }, /* Wayi Game */
	{ .dst = 5081, .app_id = { 8, 167, 0} }, /* War Thunder */
	{ .dst = 5082, .app_id = { 8, 168, 0} }, /* Warframe */
	{ .dst = 5083, .app_id = { 8, 169, 0} }, /* TT-Play Game */
	{ .dst = 5084, .app_id = { 8, 170, 0} }, /* TT-Play */
	{ .dst = 5085, .app_id = { 8, 171, 0} }, /* Robocraft */
	{ .dst = 5086, .app_id = { 8, 172, 0} }, /* World of Tanks */
	{ .dst = 5087, .app_id = { 8, 173, 0} }, /* Divinity */
	{ .dst = 5088, .app_id = { 8, 174, 0} }, /* Left 4 Dead 2 */
	{ .dst = 5089, .app_id = { 8, 175, 0} }, /* DayZ */
	{ .dst = 5090, .app_id = { 8, 176, 0} }, /* Heroes of the Storm */
	{ .dst = 5091, .app_id = { 8, 178, 0} }, /* TXWY Game */
	{ .dst = 5092, .app_id = { 8, 188, 0} }, /* Smite */
	{ .dst = 5093, .app_id = { 8, 190, 0} }, /* FreeStyle 2 Street Basketball */
	{ .dst = 5094, .app_id = { 8, 191, 0} }, /* Yeapgame Game */
	{ .dst = 5095, .app_id = { 8, 195, 0} }, /* BlackShot */
	{ .dst = 5096, .app_id = { 8, 198, 0} }, /* Combat Arms */
	{ .dst = 5097, .app_id = { 8, 202, 0} }, /* Blade and Soul */
	{ .dst = 5098, .app_id = { 8, 203, 0} }, /* FUNMILY Game */
	{ .dst = 5099, .app_id = { 8, 212, 0} }, /* Elsword */
};

/* ----- functions ------ */
uint32_t dpi_classify_app_test(struct sk_buff *skb, unsigned long *flags,
			       int lookup_flags)
{
	struct test_map *t;
	int l3proto, l4proto = 0;
	int src_port = 0, dst_port = 0;
	int i;

	if (!test_mode)
		return 0;

	l3proto = ntohs(skb->protocol);
	if (l3proto == ETH_P_IP)
		l4proto = ip_hdr(skb)->protocol;
	if (l3proto == ETH_P_IPV6)
		l4proto = ipv6_hdr(skb)->nexthdr;

	if (l4proto == IPPROTO_UDP) {
		src_port = ntohs(udp_hdr(skb)->source);
		dst_port = ntohs(udp_hdr(skb)->dest);
	} else if (l4proto == IPPROTO_TCP) {
		src_port = ntohs(tcp_hdr(skb)->source);
		dst_port = ntohs(tcp_hdr(skb)->dest);
	}

	if (test_mode == 1) {
		for (i = 0; i < ARRAY_SIZE(test_map); i++) {
			t = &test_map[i];
			if (t->src && t->src != src_port)
				continue;
			if (t->dst && t->dst != dst_port)
				continue;
			return dpi_make_app_id(t->app_id.cat,
					       t->app_id.app,
					       t->app_id.beh);
		}
	} else if (test_mode == 2) {
		t = &test_map[dst_port % ARRAY_SIZE(test_map)];
		return dpi_make_app_id(t->app_id.cat,
				       t->app_id.app,
				       t->app_id.beh);
	} else if (test_mode == 3) {
		return dst_port << 8;
	}

	return 0;
}

static int test_mode_proc_show(struct seq_file *s, void *v)
{
	seq_printf(s, "%u\n", test_mode);
	return 0;
}

static int test_mode_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, test_mode_proc_show, NULL);
}

static ssize_t
test_mode_proc_write(struct file *file, const char __user *buffer,
		     size_t count, loff_t *pos)
{
	unsigned long val;
	int ret;

	ret = kstrtoul_from_user(buffer, count, 10, &val);
	if (ret)
		return ret;

	test_mode = val;

	*pos += count;

	return count;
}

static const struct file_operations test_mode_proc_fops = {
	.owner		= THIS_MODULE,
	.open		= test_mode_proc_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release,
	.write		= test_mode_proc_write,
};


int __init dpi_test_init(void)
{
	struct proc_dir_entry *pde;

	pde = proc_create("test_mode", 0440, dpi_dir, &test_mode_proc_fops);
	if (!pde) {
		pr_err("couldn't create proc entry 'test_mode'\n");
		return -1;
	}

	return 0;
}

void dpi_test_exit(void)
{
	remove_proc_entry("test_mode", dpi_dir);
}
