#! /bin/sh

# Monitor and store core dumps of crashing processes

# default configuration
ODL_FILE=/etc/amx/amx-faultmonitor/amx-faultmonitor.odl
ODL_DEFAULT_DIR=/etc/amx/amx-faultmonitor/defaults.d/
MANAGE_FAULT_AT_SHUTDOWN=false
MANAGE_FAULT_AT_EARLY_BOOT=false
PROCESS_FAULT_BOOT=/tmp/.amx-faultmonitor.boot
COREBASEDIR=/ext/faults
MINFREESPACE=3000
MAXCOREDUMPS=20
LOCKDIR=/tmp/faultmonitor.lock
UPLOADSERVER=""
UPLOADTIMEOUT=60
MAXLOGENTRIES=100


get_config() {
    if [ -f "$ODL_FILE" ]; then
        COREBASEDIR="$(amxo-cg -s -Gjson-dm -i $ODL_DEFAULT_DIR $ODL_FILE | jsonfilter -e '@["ProcessFaults."].StoragePath')"
        MAXCOREDUMPS="$(amxo-cg -s -Gjson-dm -i $ODL_DEFAULT_DIR $ODL_FILE | jsonfilter -e '@["ProcessFaults."].MaxProcessFaultEntries')"
        MINFREESPACE="$(amxo-cg -s -Gjson-dm -i $ODL_DEFAULT_DIR $ODL_FILE | jsonfilter -e '@["ProcessFaults."].MinFreeSpace')"
        MANAGE_FAULT_AT_SHUTDOWN="$(amxo-cg -s -Gjson-cfg -i $ODL_DEFAULT_DIR $ODL_FILE | jsonfilter -e '@.manage_fault_at_shutdown')"
        MANAGE_FAULT_AT_EARLY_BOOT="$(amxo-cg -s -Gjson-cfg -i $ODL_DEFAULT_DIR $ODL_FILE | jsonfilter -e '@.manage_fault_at_early_boot')"
    fi
}

get_config

# Do not handle crash dumps if it occurs before process fault start
# and manage_fault_at_early_boot config is false
if [ ! -e "$PROCESS_FAULT_BOOT" ] && [ $MANAGE_FAULT_AT_EARLY_BOOT = "false" ]; then
    exit 0
fi

if [ -z "$COREBASEDIR" ]; then
    exit 0
fi

# Basic characteristics of the fault
PID=$1
SIGNAL=$2
TIME=$(date -u -d @$3 '+%Y-%m-%dT%H:%M:%SZ')

if [ ! -z "$4" ]; then
	PID=$4
fi

# For systems where the arguments are seperated by null bytes, translate
# those in semi-colons (;).
CMDLINE=$(tr \\0 ";" < /proc/$PID/cmdline)

# Do not handle crash dumps of ourself
if grep -q faultmonitor /proc/$PID/cmdline; then
   exit 0
fi

# Retreive software version
SOFTWAREVERSION=$(awk -F '=' '/SOFTWAREVERSION=/ {gsub(/"/, "", $2); print $2}' /etc/environment)

# Compute the crash id and store the count of crashes
mkdir -p $COREBASEDIR
while ! mkdir $LOCKDIR 2> /dev/null; do
  # poor mans busy waiting mutex
  sleep 0
done
COUNTFILE=$COREBASEDIR/count
CRASHID=0
if [ -e $COUNTFILE ]; then
  CRASHID=$(cat $COUNTFILE)
fi
COUNT=$(expr $CRASHID + 1)

# Compute local core dump path and check limits
# note: those checks are racy
COREDUMPPATH=$COREBASEDIR/dump/$CRASHID/
FREESPACE=$(\df -k $COREBASEDIR | tail -f -n 1 | awk "{ print \$4 }")
if [ $FREESPACE -lt $MINFREESPACE ]; then
    COREDUMPPATH=""
fi
if [ $MAXCOREDUMPS -eq 0 ]; then
    COREDUMPPATH=""
fi

# Store details
RET=$(ubus call ProcessFaults.ProcessFault _add "\
    {\"parameters\":\
        {\"Reason\": $SIGNAL,\
        \"ProcessName\": \"$CMDLINE\",\
        \"FaultLocation\": \"$COREDUMPPATH\",\
        \"ProcessID\": \"$PID\",\
        \"TimeStamp\": \"$TIME\",\
        \"FirmwareVersion\": \"$SOFTWAREVERSION\"}}")

# Extract the amxd-error-code
amxd_error_code=$(echo "$RET" | sed -n 's/.*"amxd-error-code": \([0-9]*\).*/\1/p')

# Do not handle crash dumps if it occurs after process fault stop
# and manage_fault_at_shutdown config is false
if [ -z "$amxd_error_code" ] && [ $MANAGE_FAULT_AT_SHUTDOWN = "false" ] && [ -e "$PROCESS_FAULT_BOOT" ]; then
    rmdir $LOCKDIR
    exit 0
fi

# Store counter/crash id
echo $COUNT > $COUNTFILE
rmdir $LOCKDIR

# Store in log
FAULTSLOG=$COREBASEDIR/log
LOGENTRIES=0
if [ -f $FAULTSLOG ]; then
    LOGENTRIES=$(wc -l $FAULTSLOG |awk '{print $1}')
    if [ $LOGENTRIES -ge $MAXLOGENTRIES ]; then
    # remove the oldest entries
    sed -i "1d" $FAULTSLOG
    fi
fi

echo "$CRASHID $PID $SIGNAL $TIME $CMDLINE" >> $FAULTSLOG

# Copy core dump only if it is successfully add to the DM or that the ProcessFaults DM isn't accessible
if [ ! -z "$COREDUMPPATH" ] && ([ "$amxd_error_code" -eq 0 ] || [ -z "$amxd_error_code" ]); then
    mkdir -p $COREDUMPPATH
    for i in wchan maps smaps fdinfo "stat*"; do
       echo cp -r /proc/$PID/$i $COREDUMPPATH
       cp -r /proc/$PID/$i $COREDUMPPATH
    done
    ls -l /proc/$PID/fd/ > $COREDUMPPATH/fd-listing.txt
    gzip -c > $COREDUMPPATH/core.gz

    json=$(echo -e "{\n\t"\
    "\"Reason\": \"$SIGNAL\",\n\t"\
    "\"ProcessName\": \"$CMDLINE\",\n\t"\
    "\"FaultLocation\": \"$COREDUMPPATH\",\n\t"\
    "\"ProcessID\": \"$PID\",\n\t"\
    "\"TimeStamp\": \"$TIME\",\n\t"\
    "\"FirmwareVersion\": \"$SOFTWAREVERSION\"\n}")

    echo "$json" > $COREDUMPPATH/config.json
fi
