#!/bin/sh

# This script collects debug information for SAH pairing and MSS issues.
# Usage: /etc/init.d/sahpairing debuginfo

IS_MASTER=false
INTERFACE="No Interface"

# Function to check br-lan status and IP connectivity
check_br_lan() {
    echo "Checking if the br-lan interface is up and has a valid IPv4 address:"
    passed=true
    # Check if br-lan interface is up
    if ! ip link show br-lan up &>/dev/null; then
        echo "Error: br-lan interface is not up or does not exist."
        passed=false
    else
        echo "br-lan interface is UP."
    fi

    # Check for IPv4 address
    ipv4_address=$(ip a show br-lan | grep -w inet | awk '{print $2}')
    if [ -z "$ipv4_address" ]; then
        echo "Error: br-lan does not have a valid IPv4 address."
        passed=false
    else
        echo "IPv4 Address: $ipv4_address"
    fi

    if $passed; then
        echo "---------------------------OK----------------------------"
    else
        echo "---------------------------NOK----------------------------"
    fi
}

# Function to get the default gateway IP and ping it
check_default_gateway() {
    echo "Checking default gateway IP and pinging it:"
    passed=false
    gateway_ip=$(ip r | grep default | awk '{print $3}')
    if [ -n "$gateway_ip" ]; then
        echo "Default gateway IP is $gateway_ip. Pinging it..."
        ping -c 4 $gateway_ip
        if [ $? -eq 0 ]; then
            echo "Ping to default gateway $gateway_ip was successful."
            passed=true
        else
            echo "Ping to default gateway $gateway_ip failed."
        fi
    else
        echo "Default gateway IP not found."
    fi
    if $passed; then
        echo "---------------------------OK----------------------------"
    else
        echo "---------------------------NOK---------------------------"
    fi
}

check_intf_status() {
    echo "Checking interface status:"
    passed=true
    # Run the command and search for Active=1
    result=$(ba-cli UpLinkMonitor.? | grep 'Active=1')

    # Check if result is empty
    if [ -z "$result" ]; then
        echo "No active interfaces found in UpLinkMonitor"
        passed = false
    else
        index=$(echo "$result" | awk -F'.' '{print $3}' | awk -F'=' '{print $1}')
        # Execute the initial command and capture the output
        interface_output=$(ba-cli UpLinkMonitor.DefaultGateway.$index.NetModelIntf? 2>/dev/null)

        # Extract the part after '=' using shell parameter expansion
        interface=${interface_output#*=}
        interface=${interface//\"/}
        INTERFACE="$interface"

        # Print the results
        echo "Found the active interface: $interface"

        # Execute the command and store the result in a variable
        int_data=$(ba-cli "$interface?")

        # Check if "Enable=1" is part of int_data
        if echo "$int_data" | grep -q "Enable=1"; then
            echo "Interface $interface is Enabled"
            # Check if "Status=1" is part of int_data
            if echo "$int_data" | grep -q "Status=1"; then
                echo "Interface $interface is Status is OK"
            else
                echo "Interface $interface is Status is NOK"
                passed=false
            fi
        else
            echo "Interface $interface is NOT Enabled"
            passed=false
        fi

        # Execute the command and store the result in a variable
        flags_data_output=$(ba-cli "${interface}Flags?" 2>/dev/null)

        # Extract the part after '=' in flags_data_output
        flags_data=${flags_data_output#*=}

        # Check if wifi
        if echo "$flags_data" | grep -q "ssid"; then
            echo "Setup in Wifi Backhaul"
        else
            # Check if ethernet
            if echo "$flags_data" | grep -q "eth"; then
                echo "Setup in Ethernet Backhaul"
            fi
        fi

        # Check if "up" is present in flags_data
        if echo "$flags_data" | grep -q "\bup\b"; then
            echo "'up' flag is present in interface $interface"
        else
            echo "'up' flag is not present in interface $interface"
            passed=false
        fi

        # Check if "uplink" is present in flags_data
        if echo "$flags_data" | grep -q "\buplink\b"; then
            echo "'uplink' flag is present in interface $interface"
        else
            echo "'uplink' flag is not present in interface $interface"
            passed=false
        fi
    fi

    if $passed; then
        echo "---------------------------OK----------------------------"
    else
        echo "---------------------------NOK---------------------------"
    fi
}

check_deviceinfo() {
    echo "Checking if DeviceInfo datamodel exists:"
    # Run the command and capture the output
    output=$(ba-cli "DeviceInfo.?0")
    passed=fail
    if echo "$output" | grep -q "not found"; then
        echo "DeviceInfo DM does not exist"
    else
        echo "DeviceInfo DM exists"
        passed=true
    fi
    if $passed; then
        echo "---------------------------OK----------------------------"
    else
        echo "---------------------------NOK---------------------------"
    fi
}

# Function to check if Master or Slave
check_mss_mode() {
    echo "Master/Slave Configuration"
    mode=$(ba-cli "MSS.Mode?")
    echo "$mode"
    if echo "$mode" | grep -q "Master"; then
        IS_MASTER=true
    else
        IS_MASTER=false
    fi
    echo "---------------------------------------------------------"
}

# Function to check MSS status
check_mss_status() {
    echo "Master/Slave Configuration"
    passed=false
    mode=$(ba-cli "MSS.Status?")
    echo "$mode"
    if $IS_MASTER; then
        if echo "$mode" | grep -q "Master"; then
            passed=true
        fi
    else
        if echo "$mode" | grep -q "Slave"; then
            passed=true
        fi
    fi
    if $passed; then
        echo "---------------------------OK----------------------------"
    else
        echo "---------------------------NOK---------------------------"
    fi
}

# Function to check SAHPairing status
check_sahpairing_status() {
    echo "Checking SAHPairing Status:"
    status=$(ba-cli "SAHPairing.Status?")
    passed=false
    echo "$status"

    if $IS_MASTER; then
        if echo "$status" | grep -q "MasterIdle"; then
            passed=true
        fi
    else
        if echo "$status" | grep -q "SlavePaired"; then
            passed=true
        fi
    fi

    if $passed; then
        echo "---------------------------OK----------------------------"
    else
        echo "---------------------------NOK---------------------------"
    fi
}

# Function to check DNSSD
check_dnssd() {
    echo "Checking DNSSD Configuration:"
    passed=true
    if $IS_MASTER; then
        output=$(ba-cli "DNSSD.Advertise.?")
        if echo "$output" | grep -q "mss_mqtt"; then
            echo "mss_mqtt advertisement found"
        else
            echo "mss_mqtt advertisement not found"
            passed=false
        fi
        if echo "$output" | grep -q "mss_ssw"; then
            echo "mss_ssw advertisement found"
        else
            echo "mss_ssw advertisement not found"
            passed=false
        fi
        if ! $passed; then
            echo "Checking umdns config files"
            cat /etc/umdns/*
        fi
    else
        output=$(ba-cli "DNSSD.Service.?")
        if echo "$output" | grep -q "master"; then
            echo "master service found"
        else
            echo "master service not found"
            passed=false
        fi

        if echo "$output" | grep -q "pair-device"; then
            echo "pair-device service found"
        else
            echo "pair-device service not found"
            passed=false
        fi
    fi

    if $passed; then
        echo "---------------------------OK----------------------------"
    else
        echo "---------------------------NOK---------------------------"
    fi
}

# Function to check umdns is running
check_umdns() {
    echo "Checking if umdns is running:"
    passed=false
    pid=$(pidof umdns)
    if [ -n "$pid" ]; then
        echo "umdns is running - PID:$pid"
        passed=true
    else
        echo "umdns not running"
    fi

    if $passed; then
        echo "---------------------------OK----------------------------"
    else
        echo "---------------------------NOK---------------------------"
    fi
}

# Function to check MQTT Client
check_mqtt_client() {
    echo "Checking MQTT Client status:"
    passed=true
    if $IS_MASTER; then
        local_status=$(ba-cli "MQTT.Client.local.Status?")
        echo "$local_status"
        if echo "$local_status" | grep -q "Connected"; then
            echo "Local MQTT Client exists and is OK"
        else
            echo "Local MQTT Client NOK"
            passed=false
        fi
    else
        remote_status=$(ba-cli "MQTT.Client.wlan_cs_remote.Status?")
        echo "$remote_status"
        if echo "$remote_status" | grep -q "Connected"; then
            echo "Remote MQTT Client exists and is OK"
        else
            echo "Remote MQTT Client NOK"
            passed=false
        fi
    fi
    if $passed; then
        echo "---------------------------OK----------------------------"
    else
        echo "---------------------------NOK---------------------------"
    fi
}

# Function to check MQTT Broker configuration
check_mqtt_broker() {
    echo "Checking MQTT Broker configuration:"
    passed=true
    secure_status=$(ba-cli "MQTTBroker.Broker.secure.Status?")
    echo "$secure_status"
    if echo "$secure_status" | grep -q "Enabled"; then
        echo "Secure MQTT Broker exists and is OK"
    else
        echo "Secure MQTT Broker NOK"
        passed=false
    fi

    local_status=$(ba-cli "MQTTBroker.Broker.local.Status?")
    echo "$local_status"
    if echo "$local_status" | grep -q "Enabled"; then
        echo "Local MQTT Broker exists and is OK"
    else
        echo "Local MQTT Broker NOK"
        passed=false
    fi
    if $passed; then
        echo "---------------------------OK----------------------------"
    else
        echo "---------------------------NOK---------------------------"
    fi
}

# Function to check SSW configuration
check_ssw() {
    echo "Checking SSW configuration:"
    passed=true
    steering_mode=$(ba-cli "SSW.Steering.Mode?")
    echo "$steering_mode"

    if $IS_MASTER; then
        if echo "$steering_mode" | grep -q "Master"; then
            echo "SSW steering mode set to Master as expected"
        else
            echo "SSW steering mode is wrong. Expected mode: Master"
            passed=false
        fi
        echo "List of Slaves:"
        ba-cli "SSW.Steering.MasterState.Slave.?"
    else
        if echo "$steering_mode" | grep -q "Slave"; then
            echo "SSW steering mode set to Slave as expected"
        else
            echo "SSW steering mode is wrong. Expected mode: Slave"
            passed=false
        fi
        conn_state=$(ba-cli "SSW.Steering.SlaveState.ConnectionState?")
        echo "$conn_state"
        if echo "$conn_state" | grep -q "Connected"; then
            echo "SSW Slave Connection State is Connected as expected"
        else

            echo "SSW Slave Connection State is wrong. Expected State: Connected"
            passed=false
        fi
    fi

    if $passed; then
        echo "---------------------------OK----------------------------"
    else
        echo "---------------------------NOK---------------------------"
    fi
}

# Function to check SSW configuration
check_wlan_manager() {
    echo "Checking WLANManager configuration:"
    passed=true
    sync_state=$(ba-cli "WLanManager.Repeater.SyncState?")
    echo "$sync_state"
    if echo "$sync_state" | grep -q "DistributedSyncActive"; then
        echo "Repeater Sync State is OK"
    else
        echo "Repeater Sync State NOK"
        passed=false
    fi
    if $passed; then
        echo "---------------------------OK----------------------------"
    else
        echo "---------------------------NOK---------------------------"
    fi
}

dump_debug() {
    echo "Dumping all debug information"
    echo "-------------------------CRASHES----------------------------"
    if [ -f "/ext/faults/log" ]; then
        cat "/ext/faults/log"
    else
        echo "No crashes found"
    fi
    echo "-------------------------NETWORK----------------------------"
    ip link show br-lan
    ip a show br-lan
    brctl show br-lan
    ip route
    echo "------------------------------------------------------------"

    echo "Relevant Datamodels:"
    if $IS_MASTER; then
        echo "----------------------MQTTBroker----------------------------"
        ba-cli 'MQTTBroker.Broker.local.?'
        ba-cli 'MQTTBroker.Broker.secure.?'
        echo "--------------------------MQTT------------------------------"
        ba-cli 'MQTT.Client.local.?'
        echo "--------------------------DNSSD-----------------------------"
    else
        echo "----------------------UpLinkMonitor-------------------------"
        ba-cli 'UpLinkMonitor.?'
        echo "------------------------NetModel----------------------------"
        ba-cli $INTERFACE?
        echo "--------------------------MQTT------------------------------"
        ba-cli 'MQTT.Client.wlan_cs_remote.?'
        echo "-----------------------WLANManager--------------------------"
        ba-cli "WLanManager.Repeater.?"
        echo "--------------------------DNSSD-----------------------------"
    fi
    ba-cli 'DNSSD.?'
    echo "-------------------------MSS--------------------------------"
    ba-cli 'MSS.?'
    echo "----------------------SAHPairing----------------------------"
    ba-cli 'SAHPairing.?'
    echo "----------------------DeviceInfo----------------------------"
    ba-cli "DeviceInfo.?0"
    echo "------------------------------------------------------------"
}

# Main function to run all checks
run_all_checks() {
    echo "Starting SAH Pairing and MSS debug information collection..."
    check_mss_mode
    check_br_lan
    if ! $IS_MASTER; then
        check_default_gateway
        check_intf_status
        check_umdns
        check_wlan_manager
    fi
    check_deviceinfo
    check_mss_status
    check_sahpairing_status
    check_dnssd
    check_mqtt_client
    if $IS_MASTER; then
        check_mqtt_broker
    fi
    check_ssw
    dump_debug
    echo "Debug information collection complete."
}

run_all_checks

exit 0